using System;
using System.Net;
using System.Data;
using System.Reflection;

using gov.va.med.vbecs.Common;
using DivTable = gov.va.med.vbecs.Common.VbecsTables.VamcDivision;
using BrRules = gov.va.med.vbecs.Common.VbecsBrokenRules.LabelPrinterConfig;

namespace gov.va.med.vbecs.BOL
{
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Stas Antropov</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>2/27/2005</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	/// <summary>
	/// The class represents label printer configuration stored in VBECS DB. 
	/// </summary>

	#endregion

	public class LabelPrinterConfig : BaseBusinessObject
	{
		private int _comPortNumber;
		private int _listenerTcpPortNumber;
		private IPAddress _listenerIPAddress;


		/// <summary>
		/// Maximal value for label printer COM port number.
		/// </summary>
		public const int MaxComPortNumber = 99;

		/// <summary>
		/// Minimal value for label printer COM.
		/// </summary>
		public const int MinComPortNumber = 1;

		/// <summary>
		/// Default value for Label printer COM port number.
		/// </summary>
		public const int DefaultComPortNumber = 4;
 
		/// <summary>
		/// Maximal value for label printer listener TCP port number. 
		/// </summary>
		public const int MaxTcpPortNumber = 65535;

		/// <summary>
		/// Minimal value for label printer listener TCP port number. 
		/// </summary>
		public const int MinTcpPortNumber = 1;

		/// <summary>
		/// Default value for label printer listener TCP port number. 
		/// </summary>
		public const int DefaultTcpPortNumber = 21777;

		
		#region constructors

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/15/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4702"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Object created.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4708"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>1 business rule is broken.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4709"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>COM port is assigned a default value.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4710"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>TCP port is assigned a default value.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4723"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Object is not marked dirty.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4703"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>None.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Default constructor.
		/// </summary>
		public LabelPrinterConfig() : base() 
		{
			SetInitialBrokenRules( BrRules.FormClassName, BrRules.RuleSets.BaseRuleSet );

			ComPortNumber = DefaultComPortNumber;
			ListenerTcpPortNumber = DefaultTcpPortNumber;
			_listenerIPAddress = null;

			IsNew = true;
			IsDirty = false;
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/16/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4765"> 
		///		<ExpectedInput>Valid DataRow with data.</ExpectedInput>
		///		<ExpectedOutput>Initialized object.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4766"> 
		///		<ExpectedInput>Valid DataRow with data.</ExpectedInput>
		///		<ExpectedOutput>Valid object not marked dirty.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4767"> 
		///		<ExpectedInput>Valid DataRow with data.</ExpectedInput>
		///		<ExpectedOutput>All fields are restored from DataRow.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4763"> 
		///		<ExpectedInput>Null.</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4764"> 
		///		<ExpectedInput>Invalid DataRow (missing columns).</ExpectedInput>
		///		<ExpectedOutput>ArgumentException.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4768"> 
		///		<ExpectedInput>DataRow containing invalid data.</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Creates an instance of a class from a supplied <see cref="DataRow"/>.
		/// </summary>
		/// <param name="dtRow"><see cref="DataRow"/> to load data from.</param>
		public LabelPrinterConfig( DataRow dtRow ) : this()
		{
			LoadFromDataRow( dtRow );
		}

		#endregion


		#region methods

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/8/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4668"> 
		///		<ExpectedInput>Empty DataRow</ExpectedInput>
		///		<ExpectedOutput>Object data saved to DataRow.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4669"> 
		///		<ExpectedInput>Null.</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4729"> 
		///		<ExpectedInput>Valid DataRow, but object is invalid.</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4730"> 
		///		<ExpectedInput>Invalid DataRow with missing columns.</ExpectedInput>
		///		<ExpectedOutput>ArgumentException.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Loads data from this instance of the class into a supplied <see cref="DataRow"/>.
		/// </summary>
		/// <param name="dtRow"><see cref="DataRow"/> to load data into.</param>
		/// <returns>Reference to provided <see cref="DataRow"/>.</returns>
		public override DataRow LoadDataRowFromThis( DataRow dtRow )
		{
			if( dtRow == null )
				throw( new ArgumentNullException( "dtRow" ) );

			Utility.RequireColumns( dtRow, DivTable.LabelPrinterComPortNumber, DivTable.LabelPrinterIPAddress, DivTable.LabelPrinterTcpPortNumber );

			if( !IsValid )
				throw( new BusinessObjectException( StrRes.SysErrMsg.Common.UnableToSaveInvalidBusinessObject( typeof( LabelPrinterConfig ).Name ).ResString ) );

			dtRow[ DivTable.LabelPrinterComPortNumber ] = StDbNullConvert.From( _comPortNumber );
			dtRow[ DivTable.LabelPrinterIPAddress ] = StDbNullConvert.From( _listenerIPAddress );
			dtRow[ DivTable.LabelPrinterTcpPortNumber ] = StDbNullConvert.From( _listenerTcpPortNumber );
			
			return dtRow;
		}

		/// <summary>
		/// Loads data into this instance of the class from a supplied <see cref="DataRow"/>.
		/// </summary>
		/// <param name="dtRow"><see cref="DataRow"/> to load data from.</param>
		protected override void LoadFromDataRow( DataRow dtRow )
		{
			if( dtRow == null )
				throw( new ArgumentNullException( "dtRow" ) );

			Utility.RequireNonNullColumns( dtRow, DivTable.LabelPrinterComPortNumber, DivTable.LabelPrinterTcpPortNumber, DivTable.LabelPrinterIPAddress );

			ComPortNumber = StDbNullConvert.ToInt32( dtRow[ DivTable.LabelPrinterComPortNumber ] );
			ListenerTcpPortNumber = StDbNullConvert.ToInt32( dtRow[ DivTable.LabelPrinterTcpPortNumber ] );
			ListenerIPAddress = StDbNullConvert.ToIPAddress( dtRow[ DivTable.LabelPrinterIPAddress ] );

			if( !IsValid )
				throw( new BusinessObjectException( StrRes.SysErrMsg.Common.InvalidDataLoadedFromDataRowIntoBusinessObject( typeof( LabelPrinterConfig ).Name, MethodBase.GetCurrentMethod().Name ).ResString ) );

			IsNew = false;
			IsDirty = false;
		}

		#endregion


		#region properties
		
		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/8/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="4325"> 
		///		<ExpectedInput>Valid test COM port number.</ExpectedInput>
		///		<ExpectedOutput>Value is set.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4326"> 
		///		<ExpectedInput>Valid test COM port number.</ExpectedInput>
		///		<ExpectedOutput>Object is marked dirty.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4718"> 
		///		<ExpectedInput>Valid test COM port number.</ExpectedInput>
		///		<ExpectedOutput>"Not set" rule is not broken.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4719"> 
		///		<ExpectedInput>Valid test COM port number.</ExpectedInput>
		///		<ExpectedOutput>"Out of range" rule is not broken.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4720"> 
		///		<ExpectedInput>Valid test COM port number equal to pre-set COM port number.</ExpectedInput>
		///		<ExpectedOutput>Object is not marked dirty.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4666"> 
		///		<ExpectedInput>Invalid COM port number value.</ExpectedInput>
		///		<ExpectedOutput>"Not set" rule is not broken.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4667"> 
		///		<ExpectedInput>Invalid COM port number value.</ExpectedInput>
		///		<ExpectedOutput>Out of range rule is broken.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4721"> 
		///		<ExpectedInput>Null value.</ExpectedInput>
		///		<ExpectedOutput>"Not set" rule is broken.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4722"> 
		///		<ExpectedInput>Null value.</ExpectedInput>
		///		<ExpectedOutput>"Out of range" rule is not broken.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Printer COM port number.
		/// </summary>
		public int ComPortNumber
		{
			get
			{
				return _comPortNumber;
			}
			set
			{
				if( _comPortNumber == value )
					return;

				RuleBroken( BrRules.ComPortNumberOutOfRange, 
					!RuleBroken( BrRules.ComPortNumberNotSet, StNullConvert.IsNull( value ) ) &&
					( value < MinComPortNumber || value > MaxComPortNumber ) );
				
				IsDirty = true;
				_comPortNumber = value;
			}
		}

		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/10/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4724"> 
		///		<ExpectedInput>Valid test IP address.</ExpectedInput>
		///		<ExpectedOutput>Value assigned.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4725"> 
		///		<ExpectedInput>Valid test IP address.</ExpectedInput>
		///		<ExpectedOutput>Object is marked dirty.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4726"> 
		///		<ExpectedInput>Valid test IP address.</ExpectedInput>
		///		<ExpectedOutput>"Not set" rule is not broken.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4728"> 
		///		<ExpectedInput>Valid IP address equal to pre-set IP.</ExpectedInput>
		///		<ExpectedOutput>Object is not marked dirty.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4727"> 
		///		<ExpectedInput>Null value.</ExpectedInput>
		///		<ExpectedOutput>"Not set" rule is broken.</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Printer listener (or printer itself) IP address.
		/// </summary>
		public IPAddress ListenerIPAddress
		{
			get
			{
				return _listenerIPAddress;
			}
			set
			{
				if( ( _listenerIPAddress == null && value == null ) ||
					( _listenerIPAddress != null && _listenerIPAddress.Equals( value ) ) )
					return;

				RuleBroken( BrRules.ListenerIPAddressNotSet, StNullConvert.IsNull( value ) );
						
				IsDirty = true;
				_listenerIPAddress = value;
			}
		}

		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/10/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4323"> 
		///		<ExpectedInput>Valid test TCP port number value.</ExpectedInput>
		///		<ExpectedOutput>Value assigned.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4712"> 
		///		<ExpectedInput>Valid test TCP port number value.</ExpectedInput>
		///		<ExpectedOutput>Object is marked dirty</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4713"> 
		///		<ExpectedInput>Valid test TCP port number value.</ExpectedInput>
		///		<ExpectedOutput>"Not set" rule is not broken.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4714"> 
		///		<ExpectedInput>Valid test TCP port number value.</ExpectedInput>
		///		<ExpectedOutput>"Out of range" rule is not broken.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4717"> 
		///		<ExpectedInput>Value equal to pre-set TCP port number value.</ExpectedInput>
		///		<ExpectedOutput>Object is not marked dirty.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4324"> 
		///		<ExpectedInput>Invalid value for TCP port number.</ExpectedInput>
		///		<ExpectedOutput>"Not set" rule is not broken.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4716"> 
		///		<ExpectedInput>Invalid value for TCP port number.</ExpectedInput>
		///		<ExpectedOutput>"Out of range" rule is broken.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4711"> 
		///		<ExpectedInput>Null value.</ExpectedInput>
		///		<ExpectedOutput>"Not set" rule is broken.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4715"> 
		///		<ExpectedInput>Null value.</ExpectedInput>
		///		<ExpectedOutput>"Out of range" rule is not broken.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Printer listener (or printer itself) TCP port number.
		/// </summary>
		public int ListenerTcpPortNumber
		{
			get
			{
				return _listenerTcpPortNumber;
			}
			set
			{
				if( _listenerTcpPortNumber == value )
					return;

				RuleBroken( BrRules.ListenerTcpPortNumberOutOfRange, 
					!RuleBroken( BrRules.ListenerTcpPortNumberNotSet, StNullConvert.IsNull( value ) ) &&
					( value < MinTcpPortNumber || value > MaxTcpPortNumber ) );

				IsDirty = true;
				_listenerTcpPortNumber = value;
			}
		}

		#endregion
	}
}
